/**
 * @file
 * Marker InfoWindow.
 */

/**
 * @typedef {Object} MarkerInfoWindowSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {Boolean} infoAutoDisplay
 * @property {Boolean} disableAutoPan
 * @property {Boolean} infoWindowSolitary
 * @property {int} maxWidth
 */

/**
 * @typedef {Object} GoogleInfoWindow
 * @property {Function} open
 * @property {Function} close
 */

/**
 * @property {GoogleInfoWindow} GeolocationGoogleMap.infoWindow
 * @property {function({}):GoogleInfoWindow} GeolocationGoogleMap.InfoWindow
 */

(function (Drupal) {
  "use strict";

  /**
   * Marker InfoWindow.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationMarkerInfoWindow = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "marker_infowindow",

        /**
         * @param {GeolocationGoogleMap} map - Current map.
         * @param {MarkerInfoWindowSettings} featureSettings - Settings for current feature.
         */
        function (map, featureSettings) {
          map.addMarkerAddedCallback(function (currentMarker) {
            if (typeof currentMarker.locationWrapper === "undefined") {
              return;
            }

            var content =
              currentMarker.locationWrapper.find(".location-content");

            if (content.length < 1) {
              return;
            }
            content = content.html();

            var markerInfoWindow = {
              content: content.toString(),
              disableAutoPan: featureSettings.disableAutoPan,
            };

            if (featureSettings.maxWidth > 0) {
              markerInfoWindow.maxWidth = featureSettings.maxWidth;
            }

            // Set the info popup text.
            var currentInfoWindow = new google.maps.InfoWindow(
              markerInfoWindow
            );

            currentMarker.addListener("click", function () {
              if (featureSettings.infoWindowSolitary) {
                if (typeof map.infoWindow !== "undefined") {
                  map.infoWindow.close();
                }
                map.infoWindow = currentInfoWindow;
              }
              currentInfoWindow.open(map.googleMap, currentMarker);
            });

            if (featureSettings.infoAutoDisplay) {
              if (map.googleMap.get("tilesloading")) {
                google.maps.event.addListenerOnce(
                  map.googleMap,
                  "tilesloaded",
                  function () {
                    google.maps.event.trigger(currentMarker, "click");
                  }
                );
              } else {
                jQuery.each(map.mapMarkers, function (index, currentMarker) {
                  google.maps.event.trigger(currentMarker, "click");
                });
              }
            }
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(Drupal);
;
/**
 * @file
 * Control locate.
 */

(function ($, Drupal) {
  "use strict";

  /**
   * Locate control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationControlLocate = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_locate",

        /**
         * @param {GeolocationMapInterface} map
         * @param {GeolocationMapFeatureSettings} featureSettings
         */
        function (map, featureSettings) {
          map.addInitializedCallback(function (map) {
            var locateButton = $(
              ".geolocation-map-control .locate",
              map.wrapper
            );

            if (
              navigator.geolocation &&
              window.location.protocol === "https:"
            ) {
              locateButton.click(function (e) {
                navigator.geolocation.getCurrentPosition(function (
                  currentPosition
                ) {
                  var currentLocation = new google.maps.LatLng(
                    currentPosition.coords.latitude,
                    currentPosition.coords.longitude
                  );
                  map.setCenterByCoordinates(
                    currentLocation,
                    currentPosition.coords.accuracy,
                    "google_control_locate"
                  );
                });
                e.preventDefault();
              });
            } else {
              locateButton.remove();
            }
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(jQuery, Drupal);
;
/**
 * @file
 * Zoom Control.
 */

/**
 * @typedef {Object} ControlZoomSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} behavior
 * @property {String} position
 * @property {String} style
 */

(function (Drupal) {
  "use strict";

  /**
   * Zoom control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationZoomControl = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_zoom",

        /**
         * @param {GeolocationGoogleMap} map - Current map.
         * @param {ControlZoomSettings} featureSettings - Settings for current feature.
         */
        function (map, featureSettings) {
          map.addPopulatedCallback(function (map) {
            var options = {
              zoomControlOptions: {
                position: google.maps.ControlPosition[featureSettings.position],
                style: google.maps.ZoomControlStyle[featureSettings.style],
              },
            };

            if (featureSettings.behavior === "always") {
              options.zoomControl = true;
            } else {
              options.zoomControl = undefined;
            }
            map.googleMap.setOptions(options);
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(Drupal);
;
/**
 * @file
 * Control MapType.
 */

/**
 * @typedef {Object} ControlMapTypeSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} position
 * @property {String} style
 * @property {String} behavior
 */

(function (Drupal) {
  "use strict";

  /**
   * Maptype control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationMapTypeControl = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_maptype",

        /**
         * @param {GeolocationGoogleMap} map - Current map.
         * @param {ControlMapTypeSettings} featureSettings - Settings for current feature.
         */
        function (map, featureSettings) {
          map.addPopulatedCallback(function (map) {
            var options = {
              mapTypeControlOptions: {
                position: google.maps.ControlPosition[featureSettings.position],
                style: google.maps.MapTypeControlStyle[featureSettings.style],
              },
            };

            if (featureSettings.behavior === "always") {
              options.mapTypeControl = true;
            } else {
              options.mapTypeControl = undefined;
            }

            map.googleMap.setOptions(options);
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(Drupal);
;
{const a=t=>t?typeof t[Symbol.iterator]=="function"?[...t]:[t]:[],f=(t,s)=>{const r=["fontSize"];return s&&r.push("lineHeight"),(o,l=e=>e)=>{t.forEach(e=>{r.filter(n=>n in o).forEach(n=>{e.style[n]=o[n]?`${l(o[n])}px`:""})})}},c=new WeakMap,h=t=>{const s=t<1?"min":"max",r=t<1?Math.max:Math.min;return({fontSize:o,lineHeight:l},e)=>{const n=e[`text_resize_${s}imum`],i={fontSize:r(o*t,n)};return e.text_resize_line_height_allow&&(i.lineHeight=i.fontSize===n?e[`text_resize_line_height_${s}`]:l*t),localStorage.setItem("textResize",JSON.stringify(i)),i}},z={text_resize_increase:h(1.2),text_resize_decrease:h(1/1.2),text_resize_reset:(t,s)=>{const r={fontSize:null};return s.text_resize_line_height_allow&&(r.lineHeight=null),localStorage.removeItem("textResize"),r}};Drupal.behaviors.textResize={attach(t,{text_resize:s={}}){const{text_resize_scope:r,text_resize_line_height_allow:o}=s;let l=[];if(r?l=["getElementById","getElementsByClassName","querySelectorAll"].reduce((e,n)=>e.length?e:a(document[n](r)),l):l=["page","content-inner","#squeeze > #content"].reduce((e,n,i)=>e.length?e:a(document[i===2?"querySelectorAll":"getElementById"](n)),l),l.length){const e=f(l,o);if(typeof localStorage.textResize<"u")try{e(JSON.parse(localStorage.textResize),parseFloat)}catch{}const n=i=>{i.preventDefault();const{currentTarget:g}=i,_=getComputedStyle(l[0]),x=z[g.id]({fontSize:parseFloat(_.fontSize),lineHeight:parseFloat(_.lineHeight)},s);e(x)};once("text-resize","a.changer").forEach(i=>{c.set(i,n),i.addEventListener("click",n)})}},detach(t,s,r){r==="unload"&&once.filter("text-resize","a.changer",t).filter(o=>c.has(o)).forEach(o=>o.removeEventListener("click",c.get(o)))}}}
;
/**
* DO NOT EDIT THIS FILE.
* See the following change record for more information,
* https://www.drupal.org/node/2815083
* @preserve
**/
(function (Drupal, drupalSettings) {
  function mapTextContentToAjaxResponse(content) {
    if (content === '') {
      return false;
    }
    try {
      return JSON.parse(content);
    } catch (e) {
      return false;
    }
  }
  function bigPipeProcessPlaceholderReplacement(placeholderReplacement) {
    var placeholderId = placeholderReplacement.getAttribute('data-big-pipe-replacement-for-placeholder-with-id');
    var content = placeholderReplacement.textContent.trim();
    if (typeof drupalSettings.bigPipePlaceholderIds[placeholderId] !== 'undefined') {
      var response = mapTextContentToAjaxResponse(content);
      if (response === false) {
        once.remove('big-pipe', placeholderReplacement);
      } else {
        var ajaxObject = Drupal.ajax({
          url: '',
          base: false,
          element: false,
          progress: false
        });
        ajaxObject.success(response, 'success');
      }
    }
  }
  var interval = drupalSettings.bigPipeInterval || 50;
  var timeoutID;
  function bigPipeProcessDocument(context) {
    if (!context.querySelector('script[data-big-pipe-event="start"]')) {
      return false;
    }
    once('big-pipe-early-behaviors', 'body', context).forEach(function (el) {
      Drupal.attachBehaviors(el);
    });
    once('big-pipe', 'script[data-big-pipe-replacement-for-placeholder-with-id]', context).forEach(bigPipeProcessPlaceholderReplacement);
    if (context.querySelector('script[data-big-pipe-event="stop"]')) {
      if (timeoutID) {
        clearTimeout(timeoutID);
      }
      return true;
    }
    return false;
  }
  function bigPipeProcess() {
    timeoutID = setTimeout(function () {
      if (!bigPipeProcessDocument(document)) {
        bigPipeProcess();
      }
    }, interval);
  }
  bigPipeProcess();
  window.addEventListener('load', function () {
    if (timeoutID) {
      clearTimeout(timeoutID);
    }
    bigPipeProcessDocument(document);
  });
})(Drupal, drupalSettings);;
